<?php

namespace backend\controllers;

use Yii;
use common\models\Registrations;
use backend\models\RegistrationsSearch;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use common\models\User;
use common\components\CsvExport;
/**
 * RegistrationController implements the CRUD actions for registrations model.
 */
class RegistrationController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => \yii\filters\AccessControl::className(),
                'only' => ['index', 'create', 'update', 'delete', 'view', 'export', 'bulk'],
                'rules' => [
                    // deny all POST requests
                    // [
                    //     'allow' => false,
                    //     'verbs' => ['POST']
                    // ],
                    // allow authenticated users
                    [
                        'actions' => ['index', 'create', 'update', 'delete', 'view', 'export', 'bulk'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    // [
                    //     'actions' => ['create', 'search'],
                    //     'allow' => true,
                    //     'roles' => ['?'],
                    // ],
                    // everything else is denied
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Lists all registrations models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new RegistrationsSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }
    /**
     * Bulk actions for gridview
     * @return mixed
     */
    public function actionBulk(){
        $action = Yii::$app->request->post('action');
        $selection = (array)Yii::$app->request->post('selection');//typecasting
        
        if($action == 'resend-emails'){
            foreach($selection as $id){
                $reg = Registrations::findOne((int)$id);
                if(isset($reg)){
                    //send email to user
                    Yii::$app
                    ->mailer
                    ->compose(
                        ['html' => 'registration-html', 'text' => 'registration-text'],
                        ['model' => $reg]
                    )
                    ->setFrom([Yii::$app->params['supportEmail'] => Yii::$app->name])
                    ->setTo($reg->user->email)
                    //->setCc(['saq1b.zaf4r@gmail.com'])
                    ->SetBcc(['registration@federalservicedogregistration.org','portal@federalservicedogregistration.org'])
                    ->setSubject($reg->dog_name.'\'s Registration # = '.$reg->registration_no.' (Federal Service Dog Registration)')
                    ->send();
                }
            }
        }

        return $this->redirect(['index']);
    }

    /**
     * Export CSV function
     * @return mixed 
     */
    public function actionExport()
    {
        $model = new \backend\models\ExportForm();
        $post = Yii::$app->request->post();

        //initialize the dates
        if(empty($model->dateFrom) && empty($model->dateTo)){
            $model->dateFrom = date('d-M-Y', strtotime('Today'));
            $model->dateTo = date('d-M-Y', strtotime('Tomorrow'));
        }

        if ($model->load($post)) {
            if ($model->validate()) {
                $dateFrom = date('U', strtotime($model->dateFrom));
                $dateTo = date('U', strtotime($model->dateTo. ' +1 day'));

                switch ($model->type_of_registration) {
                    case 'All':
                        $registrations = Registrations::find()
                            ->where(['between', 'created_at', $dateFrom, $dateTo])
                            ->All();
                        break;
                    case 'Emotional Support Animal':
                        $registrations = Registrations::find()
                            ->where(['between', 'created_at', $dateFrom, $dateTo])
                            ->andWhere(['type_of_registration' => ['Emotional Support Animal', 'Emotional Support Dog']])
                            ->All();
                        break;
                    default:
                        $registrations = Registrations::find()
                        ->where(['between', 'created_at', $dateFrom, $dateTo])
                        ->andWhere(['type_of_registration' => $model->type_of_registration])
                        ->All();
                        break;
                }

                if(count($registrations) > 0){
                    CsvExport::export($registrations);
                    exit;
                } else {
                    Yii::$app->session->setFlash('error', 'No records found.');
                }
                
            } else {
                Yii::$app->session->setFlash('error', 'Please provide correct date range and type value.');
            }
        }

        return $this->render('export', [
            'model' => $model,
        ]);
    }

    /**
     * Displays a single registrations model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new registrations model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Registrations();
        $model->scenario = Registrations::SCENARIO_CREATE;

        $post = Yii::$app->request->post();


        if(!empty($post)){

            $load = $model->load($post);

            //check if user exists
            $oldUser = User::findOne(['email' => $post['Registrations']['email']]);
            if(!$oldUser){
                $user = new User();
                $user->username = $post['Registrations']['email'];
                $user->email = $post['Registrations']['email'];
                $user->setPassword($post['Registrations']['password']);
                $user->generateAuthKey();
                if(!$user->save())
                    throw new \yii\web\HttpException(500, 'Error creating user, please contact the administrator.');
                    
            }
            else{
                $user = $oldUser;
            }

            if ($load) {
                $model->user_id = $user->id;
                $model->registration_no = $model->generateUniqueRandomString();
                $model->user_ip = Yii::$app->getRequest()->getUserIP();
                //$model->created_at = time();
                //$model->updated_at = time();
            }

            if($model->save(false)){
                //send email to user
                Yii::$app
                ->mailer
                ->compose(
                    ['html' => 'registration-html', 'text' => 'registration-text'],
                    ['model' => $model]
                )
                ->setFrom([Yii::$app->params['supportEmail'] => Yii::$app->name])
                ->setTo($user->email)
                //->setCc(['saq1b.zaf4r@gmail.com'])
                ->SetBcc(['registration@federalservicedogregistration.org','portal@federalservicedogregistration.org'])
                ->setSubject($model->dog_name.'\'s Registration # = '.$model->registration_no.' (Federal Service Dog Registration)')
                ->send();
                
                return $this->redirect(['index']);
            }
                
        }

        return $this->render('create', [
            'model' => $model,
        ]);
    }

    /**
     * Updates an existing registrations model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $model->scenario = Registrations::SCENARIO_UPDATE;

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
        ]);
    }

    /**
     * Deletes an existing registrations model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the registrations model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return registrations the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Registrations::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
    }
}
